---------------------------------------------------
-- Postgrest Configuration
---------------------------------------------------

-- Funktion um die Konfiguration und oder den Schema Cache von PostgREST neuzuladen
CREATE OR REPLACE FUNCTION api.postgrest__reload(
      _db_channel            varchar   DEFAULT 'pgrst'
    , _reload_config         boolean   DEFAULT true
    , _reload_schema_cache   boolean   DEFAULT true
  ) RETURNS void AS $$
  DECLARE
      _sql                  varchar   DEFAULT 'NOTIFY ' || _db_channel;
  BEGIN
      IF _reload_config IS true AND _reload_schema_cache IS false THEN
          _sql := _sql || ', ''reload config''';
      ELSIF _reload_config IS false AND _reload_schema_cache IS true THEN
          _sql := _sql || ', ''reload schema''';
      ELSIF _reload_config IS false AND _reload_schema_cache IS false THEN
          RETURN;
      END IF;
      _sql := _sql || ';';
      --
      EXECUTE _sql;
  END $$ LANGUAGE plpgsql STRICT;

-- Funktion für die Konfiguration von PostgREST, wird von PostgREST aufgerufen (per db-pre-config in postgrest.conf)
CREATE OR REPLACE FUNCTION api.postgrest_pre_config()
  RETURNS void
  AS $$
    -- Konfiguration für PostgREST
    -- https://postgrest.org/en/stable/configuration.html
  SELECT
      -- https://postgrest.org/en/stable/api-reference.html#db-extra-search-path
      set_config('pgrst.db_extra_search_path', replace(current_setting('search_path'), '"$user", ',''), true);

  $$ LANGUAGE sql;


-- Funktion zur Prüfung der Berechtigung des Angemeldeten Benutzers, wird von PostgREST aufgerufen (per db-pre-request in postgrest.conf)
-- Wird gestartet nach der Benutzeranmeldung und vor der Ausführung der eigentlichen Restabfrage.
-- Authentication in PostgREST -> Custom Validation
-- https://postgrest.org/en/stable/references/auth.html#custom-validation
CREATE OR REPLACE FUNCTION api.postgrest_pre_request() RETURNS void AS $$
  DECLARE
      role text := current_setting('request.jwt.claims', true)::json->>'role';
      schema text := current_setting('request.headers', true)::json->>'accept-profile';
  BEGIN
      IF role = 'sys_postgrest_anon' AND schema = 'public' THEN
          RAISE EXCEPTION 'Anonymous access to schema "public" forbidden!'
              USING HINT = 'Login with valid role to access schema "public".';
      END IF;
  END $$ LANGUAGE plpgsql;


---------------------------------------------------
-- Automatischer Reload des PostgREST-Schema bei DDL-Änderungen
---------------------------------------------------

-- Event-Trigger für DDL-Änderungen im Schema 'API'
CREATE OR REPLACE FUNCTION api.pgrst_ddl_watch()
  RETURNS event_trigger
  AS $$
  DECLARE
    cmd record;
  BEGIN
    FOR cmd IN SELECT * FROM pg_event_trigger_ddl_commands()
    LOOP
      IF cmd.command_tag IN (
        'CREATE SCHEMA', 'ALTER SCHEMA'
      , 'CREATE TABLE', 'CREATE TABLE AS', 'SELECT INTO', 'ALTER TABLE'
      , 'CREATE FOREIGN TABLE', 'ALTER FOREIGN TABLE'
      , 'CREATE VIEW', 'ALTER VIEW'
      , 'CREATE MATERIALIZED VIEW', 'ALTER MATERIALIZED VIEW'
      , 'CREATE FUNCTION', 'ALTER FUNCTION'
      , 'CREATE TRIGGER'
      , 'CREATE TYPE', 'ALTER TYPE'
      , 'CREATE RULE'
      , 'COMMENT'
      )
      AND cmd.schema_name IS DISTINCT FROM 'pg_temp'
      AND cmd.schema_name ILIKE 'API'  -- Einschränkung auf API-Schema
      THEN
        PERFORM api.postgrest__reload( _reload_config => false ); -- NOTIFY pgrst, 'reload schema';
      END IF;
    END LOOP;
  END;
  $$ LANGUAGE plpgsql;

  DROP EVENT TRIGGER IF EXISTS pgrst_ddl_watch;
  CREATE EVENT TRIGGER pgrst_ddl_watch
    ON ddl_command_end
    EXECUTE PROCEDURE api.pgrst_ddl_watch();

-- Event-Trigger für DROP-Änderungen im Schema 'API'
CREATE OR REPLACE FUNCTION api.pgrst_drop_watch()
  RETURNS event_trigger
  AS $$
  DECLARE
    obj record;
  BEGIN
    FOR obj IN SELECT * FROM pg_event_trigger_dropped_objects()
    LOOP
      IF obj.object_type IN (
        'schema'
      , 'table'
      , 'foreign table'
      , 'view'
      , 'materialized view'
      , 'function'
      , 'trigger'
      , 'type'
      , 'rule'
      )
      AND obj.is_temporary IS FALSE
      AND obj.object_identity ILIKE 'API.%'  -- Einschränkung auf Objekte im Schema api
      THEN
        PERFORM api.postgrest__reload( _reload_config => false ); -- NOTIFY pgrst, 'reload schema';
      END IF;
    END LOOP;
  END;
  $$ LANGUAGE plpgsql;

  DROP EVENT TRIGGER IF EXISTS pgrst_drop_watch;
  CREATE EVENT TRIGGER pgrst_drop_watch
    ON sql_drop
    EXECUTE PROCEDURE api.pgrst_drop_watch();



---------------------------------------------------
-- File-Handling über die API
---------------------------------------------------


-- Funktion ermittelt den Content-Type einer Datei basierend auf dem Header
CREATE OR REPLACE FUNCTION tsystem.bytea__content_type__get(
    data      bytea,
    file_name text DEFAULT null
  )
  RETURNS text
  AS $$
  DECLARE
    header BYTEA;
    file_extension TEXT;
    data_text TEXT;
  BEGIN
    -- Nur die ersten 16 Bytes für die Header-Prüfung nehmen
    IF length(data) >= 16 THEN
        header := substring(data FROM 1 FOR 16);
    ELSE
        header := data; -- Falls die Daten weniger als 16 Bytes haben
    END IF;

    -- Dateiendung extrahieren, falls ein Dateiname angegeben wurde
    IF file_name IS NOT NULL THEN
        file_extension := lower(substring(file_name from E'\\.([^\\.]+)$'));
    ELSE
        file_extension := NULL;
    END IF;

    -- Magic Numbers prüfen für häufige Formate
    IF substr(encode(header, 'hex'), 1, 16) = '89504e470d0a1a0a' THEN
        RETURN 'image/png';
    ELSIF substr(encode(header, 'hex'), 1, 12) = 'ffd8ff' THEN
        RETURN 'image/jpeg';
    ELSIF substr(encode(header, 'hex'), 1, 8) = '47494638' THEN
        RETURN 'image/gif';
    ELSIF substr(encode(header, 'hex'), 1, 8) = '25504446' THEN
        RETURN 'application/pdf';
    ELSIF substr(encode(header, 'hex'), 1, 8) = '504b0304' THEN
        -- ZIP-basierte Formate
        IF file_extension = 'docx' THEN
            RETURN 'application/vnd.openxmlformats-officedocument.wordprocessingml.document';
        ELSIF file_extension = 'xlsx' THEN
            RETURN 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet';
        ELSIF file_extension = 'pptx' THEN
            RETURN 'application/vnd.openxmlformats-officedocument.presentationml.presentation';
        ELSE
            RETURN 'application/zip';
        END IF;
    ELSIF substr(encode(header, 'hex'), 1, 8) = '1f8b0800' THEN
        RETURN 'application/gzip';
    ELSIF substr(encode(header, 'hex'), 1, 12) = '377abcaf271c' THEN
        RETURN 'application/x-7z-compressed';
    ELSIF substr(encode(header, 'hex'), 1, 4) = 'edab' THEN
        RETURN 'application/x-rpm';
    ELSIF substr(encode(header, 'hex'), 1, 16) = '3c3f786d6c2076657273' THEN
        -- XML-basierte Formate
        IF file_extension = 'svg' THEN
            RETURN 'image/svg+xml';
        ELSE
            RETURN 'application/xml';
        END IF;
    ELSIF substr(encode(header, 'hex'), 1, 8) = '7b226964' OR substr(encode(header, 'hex'), 1, 8) = '7b0a2020' THEN
        RETURN 'application/json';
    END IF;

    -- Textbasierte Formate erkennen - konvertiere byte zu text und prüfe
    BEGIN
        -- Versuche zu konvertieren (kann bei binären Daten fehlschlagen)
        data_text := convert_from(data, 'UTF8');


        IF file_extension = 'html' OR file_extension = 'htm' THEN
            RETURN 'text/html';
        ELSIF file_extension = 'css' THEN
            RETURN 'text/css';
        ELSIF file_extension = 'js' THEN
            RETURN 'application/javascript';
        ELSIF file_extension = 'csv' THEN
            RETURN 'text/csv';
        ELSIF file_extension = 'txt' THEN
            RETURN 'text/plain';
        ELSIF file_extension = 'md' THEN
            RETURN 'text/markdown';
        ELSE
            RETURN 'text/plain';
        END IF;
    EXCEPTION
        WHEN OTHERS THEN
            -- Wenn die Konvertierung fehlschlägt, ist es kein Text
            NULL;
    END;

    -- Fallback für unbekannte Dateitypen
    RETURN 'application/octet-stream';
  END;
  $$ LANGUAGE plpgsql;

-- Funktion zum Ausliefern von Dateien (z.B. Bilder, JS, CSS) aus der DB oder dem Filesystem
CREATE OR REPLACE FUNCTION api.file(
    id   varchar DEFAULT NULL,
    path varchar DEFAULT NULL
  )
  RETURNS "*/*"
  SECURITY DEFINER
  AS $$
  DECLARE
    headers   text;
    blob      bytea;
    base_path varchar := REPLACE( current_setting( 'data_directory' ), 'data', 'postgrest/www/'); -- Basis-Pfad für die Dateien
    filename  varchar;
  BEGIN

    IF id IS NULL AND path IS NULL THEN
      RAISE sqlstate 'PT400' USING
        message = 'BAD REQUEST',
        detail = 'File id or path is required',
        hint = 'Please provide a file id or path';
    ELSIF id IS NOT NULL AND path IS NOT NULL THEN
      RAISE sqlstate 'PT400' USING
        message = 'BAD REQUEST',
        detail = 'File id and path are mutually exclusive',
        hint = 'Please provide a file id or path, but not both';
    END IF;


    IF id IS NOT NULL THEN
      -- File aus DB laden
      --- Header zusammenbauen
      SELECT json_build_array(
               json_build_object( 'Content-Type', files.type ),
               json_build_object( 'Content-Disposition', format( 'inline; filename="%s"', files.name ) ),
               json_build_object( 'Cache-Control', 'max-age=43200' ) -- 43200s => 12 Stunden | Reload lässt per STRG + F5 im Browser jederzeit anstoßen (MacOS: Umschalt + F5)
             )::text
        INTO headers
        FROM api.files where files.id = file.id;

      --- Header setzen
      PERFORM set_config( 'response.headers', headers, true );

      --- Blob laden
      SELECT files.file FROM api.files WHERE files.id = file.id INTO blob;

    ELSIF path IS NOT NULL THEN
      -- File aus Filesystem laden
      --- Blob laden
      SELECT pg_read_binary_file( base_path || path ) INTO blob;

      --- Filename ermitteln
      filename := substring( base_path || path FROM E'([^/\\\\]+)$');

      --- Header zusammenbauen
      SELECT json_build_array(
               json_build_object( 'Content-Type', tsystem.bytea__content_type__get( blob, filename ) ),
               json_build_object( 'Content-Disposition', format( 'inline; filename="%s"', filename ) ),
               json_build_object( 'Cache-Control', 'max-age=43200' ) -- 43200 => 12 Stunden | Reload lässt per STRG + F5 im Browser jederzeit anstoßen (MacOS: Umschalt + F5)
             )::text
        INTO headers;

      --- Header setzen
      PERFORM set_config( 'response.headers', headers, true );

    END IF;

    IF blob IS NOT null
    THEN RETURN (blob);
    ELSE raise sqlstate 'PT404' USING
      message = 'NOT FOUND',
      detail = 'File not found',
      hint = format( '%s seems to be an invalid file id', file.id );
    END IF;

  END $$ LANGUAGE plpgsql;


------------------------------------------------------
-- Funktionen für die Authentifizierung
------------------------------------------------------
-- Funktionen für Erstellung und Prüfung von JWT-Token von hier:
-- https://github.com/michelp/pgjwt

-- PBKDF2 key derivation function für postgres’ SCRAM-SHA-256 password hashes
CREATE OR REPLACE FUNCTION api_basic_auth.pbkdf2(
    salt           bytea,
    pw             text,
    count          integer,
    desired_length integer,
    algorithm      text
  )
  RETURNS bytea
  AS $$
  DECLARE
    hash_length integer;
    block_count integer;
    output      bytea;
    the_last    bytea;
    xorsum      bytea;
    i_as_int32  bytea;
    i           integer;
    j           integer;
    k           integer;
  BEGIN
    algorithm := lower(algorithm);
    CASE algorithm
    WHEN 'md5' then
      hash_length := 16;
    WHEN 'sha1' then
      hash_length := 20;
    WHEN 'sha256' then
      hash_length := 32;
    WHEN 'sha512' then
      hash_length := 64;
    ELSE
      RAISE EXCEPTION 'Unknown algorithm "%"', algorithm;
    END CASE;
    --
    block_count := ceil(desired_length::real / hash_length::real);
    RAISE NOTICE '%', block_count;
    --
    FOR i in 1 .. block_count LOOP
      i_as_int32 := E'\\000\\000\\000'::bytea || chr(i)::bytea;
      i_as_int32 := substring(i_as_int32, length(i_as_int32) - 3);
      --
      the_last := salt::bytea || i_as_int32;
      --
      xorsum := HMAC(the_last, pw::bytea, algorithm);
      the_last := xorsum;
      --
      FOR j IN 2 .. count LOOP
        the_last := HMAC(the_last, pw::bytea, algorithm);

        -- xor the two
        FOR k IN 1 .. length(xorsum) LOOP
          xorsum := set_byte(xorsum, k - 1, get_byte(xorsum, k - 1) # get_byte(the_last, k - 1));
        END LOOP;
      END LOOP;
      --
      IF output IS NULL THEN
        output := xorsum;
      ELSE
        output := output || xorsum;
      END IF;
    END LOOP;
    --
    RETURN substring(output FROM 1 FOR desired_length);
  END $$ LANGUAGE plpgsql IMMUTABLE;


-- Funktionen zum signieren von JWT
CREATE OR REPLACE FUNCTION api_basic_auth.url_encode(
    data bytea
  )
  RETURNS text
  AS $$
    SELECT translate( encode( data, 'base64' ), E'+/=\n', '-_' );
  $$ LANGUAGE sql IMMUTABLE;


-- decode base64url encoded data
CREATE OR REPLACE FUNCTION api_basic_auth.url_decode(
    data text
  )
  RETURNS bytea
  LANGUAGE sql
  AS $$
  WITH t AS (SELECT translate(data, '-_', '+/') AS trans),
     rem AS (SELECT length(t.trans) % 4 AS remainder FROM t) -- compute padding size
    SELECT decode(
        t.trans ||
        CASE WHEN rem.remainder > 0
           THEN repeat('=', (4 - rem.remainder))
           ELSE '' END,
    'base64') FROM t, rem;
  $$ IMMUTABLE;


-- sign the data with the given algorithm
CREATE OR REPLACE FUNCTION api_basic_auth.algorithm_sign(
    signables text,
    secret text,
    algorithm text
  )
  RETURNS text
  AS $$
    WITH
      alg AS (
        SELECT CASE
          WHEN algorithm = 'HS256' THEN 'sha256'
          WHEN algorithm = 'HS384' THEN 'sha384'
          WHEN algorithm = 'HS512' THEN 'sha512'
          ELSE '' END AS id )  -- hmac throws error
    SELECT api_basic_auth.url_encode( hmac( signables, secret, alg.id ) ) FROM alg;
  $$ LANGUAGE sql IMMUTABLE;


-- sign the payload with the given algorithm
CREATE OR REPLACE FUNCTION api_basic_auth.sign(
    payload json,
    secret text,
    algorithm text DEFAULT 'HS256'
  )
  RETURNS text
  AS $$
    WITH
      header AS (
        SELECT api_basic_auth.url_encode( convert_to( '{"alg":"' || algorithm || '","typ":"JWT"}', 'utf8' ) ) AS data
        ),
      payload AS (
        SELECT api_basic_auth.url_encode( convert_to( payload::text, 'utf8' ) ) AS data
        ),
      signables AS (
        SELECT header.data || '.' || payload.data AS data FROM header, payload
        )
    SELECT
        signables.data || '.' ||
        api_basic_auth.algorithm_sign( signables.data, secret, algorithm )
      FROM signables;
  $$ LANGUAGE sql IMMUTABLE;


-- Cast string als double ohne exception
CREATE OR REPLACE FUNCTION api_basic_auth.try_cast_double(
    inp text
  )
  RETURNS double precision
  AS $$
  BEGIN
    RETURN inp::double precision;
  EXCEPTION
    WHEN OTHERS THEN RETURN NULL;
  END $$ language plpgsql IMMUTABLE;


-- Funktion die den JWT-Token prüft, validiert und die Header + Payload zurückgibt
CREATE OR REPLACE FUNCTION api_basic_auth.jwt_verify(
    token text,
    secret text,
    algorithm text DEFAULT 'HS256'
  )
  RETURNS table( header json, payload json, valid boolean )
  AS $$

    SELECT jwt.header AS header
         , jwt.payload AS payload
         , jwt.signature_ok AND tstzrange(
             to_timestamp( api_basic_auth.try_cast_double( jwt.payload ->> 'nbf' ) ),
             to_timestamp( api_basic_auth.try_cast_double( jwt.payload ->> 'exp' ) )
           ) @> CURRENT_TIMESTAMP AS valid
      FROM (
            SELECT convert_from( api_basic_auth.url_decode( r[1] ), 'utf8' )::json AS header
                 , convert_from( api_basic_auth.url_decode( r[2] ), 'utf8' )::json AS payload
                 , r[3] = api_basic_auth.algorithm_sign( r[1] || '.' || r[2], secret, algorithm ) AS signature_ok
              FROM regexp_split_to_array( token, E'\\.' ) r
           ) jwt

  $$ LANGUAGE sql IMMUTABLE;


-- Mit Logindaten einen Token für die API-Nutzung abrufen und mit Gültigkeitszeitraum in Datenbank ablegen
CREATE OR REPLACE FUNCTION x_10_interfaces.api_token_generate(
    _anbieter  varchar,                       -- z.B. CERPRO, PRODAT-ERP_POSTGREST
    _user      varchar  DEFAULT current_user, -- Benutzername für den Token (optional, falls nicht angegeben wird der aktuelle Benutzer verwendet)
    _overwrite boolean  DEFAULT false,        -- true = zwingend einen neuen Token abrufen
    _valid     interval DEFAULT '1 hour',     -- Mindestgültigkeitsdauer des Tokens, falls nicht angegeben wird der Standardwert verwendet
    _params    jsonb    DEFAULT '{}'::jsonb   -- Zusätzliche Parameter für die API-Anfrage (z.B. für PRODAT-ERP_POSTGREST: {"ll_minr": 815})
  )
  RETURNS varchar
  AS $$
  DECLARE
    _api_token_record  record;
    _url               varchar;
    _content           varchar;
    _content_type      varchar;
    _response          jsonb;
    _token             varchar;
    _expires           timestamp;
  BEGIN

    -- Prüfen ob noch ein gültiger Token vorhanden ist
    SELECT token.* INTO _api_token_record
      FROM x_10_interfaces.api_token AS token
     WHERE api_name = _anbieter
       AND api_user = _user
       AND api_params = _params
       AND api_tokenexpires > now() + _valid
     ORDER BY api_tokenexpires DESC
     LIMIT 1;

    -- Falls gültiger Token gefunden wurde und overwrite nicht gesetzt, diesen zurückgeben. Sonst Hinweis und neuen Token generieren
    IF _api_token_record.api_token IS NOT NULL AND NOT _overwrite THEN
      RETURN _api_token_record.api_token;
    ELSE
      RAISE NOTICE 'kein gültiger Token gefunden, es wird versucht einen neuen abzurufen';
    END IF;

    IF _anbieter = 'CERPRO' THEN
      _url          := 'https://platform.cerpro.io/api/auth/login';
      _content_type := 'application/json';
      _content      := jsonb_build_object(
                        'email', TSystem.Settings__GetText( 'CERPRO_EMAIL' ),
                        'password', TSystem_security.credential__get( servicename => _anbieter, username => TSystem.Settings__GetText( 'CERPRO_EMAIL' ) ) -- SELECT TSystem_security.credential__set( servicename => _anbieter, username => TSystem.Settings__GetText( 'CERPRO_EMAIL' ), userpassword => '<geheimes passwort>', overwrite_pw => true );
                      )::varchar;

      -- Sende den HTTP POST Request an die API um den Token abzurufen
      SELECT content::jsonb INTO _response
        FROM tsystem.http_post_request( _url, _content, _content_type );

      -- Token und Ablaufzeit ermitteln
      IF _response IS not null THEN
        _token   := _response->>'AccessToken';
        _expires := now() + ( ( _response->>'ExpiresIn' )::numeric || ' seconds' )::interval;
      ELSE
        RAISE EXCEPTION 'Fehler beim Abrufen eines neuen Tokens. URL: %s | Content: %s | Content-Type: %s', _url, _content, _content_type;
      END IF;

    ELSIF _anbieter = 'PRODAT-ERP_POSTGREST' THEN
      -- API-Token für PRODAT-ERP_POSTGREST generieren wir selbst per JWT
      _expires := now() + '1 day'::interval ;

      SELECT api_basic_auth.sign(
               row_to_json(r), TSystem.Settings__Database__Get('pgrst.jwt_secret')
             ) AS token
        FROM ( SELECT _user AS role
                  , COALESCE(
                      (_params ->> 'll_minr')::int,
                      (SELECT ll_minr FROM llv WHERE ll_db_usename = _user) -- Mitarbeiter-Nummer ermitteln falls nicht übergeben
                    ) AS ll_minr
                  , extract( epoch FROM _expires )::integer AS exp -- 24*60*60 s => 24h
             ) r
        INTO _token;

    ELSE
      RAISE EXCEPTION 'API-Anbieter % ist nicht implementiert.', _anbieter;
    END IF;

    -- Ablegen des Token
    IF _token IS NOT null AND _expires IS NOT null THEN
      INSERT INTO x_10_interfaces.api_token ( api_name, api_user, api_token, api_params, api_tokenexpires )
           VALUES ( _anbieter, _user, _token, _params, _expires )
      ON CONFLICT ( api_name, api_user, api_params ) DO UPDATE
              SET api_token = EXCLUDED.api_token, api_tokenexpires = EXCLUDED.api_tokenexpires;
    ELSE
      RAISE EXCEPTION 'Fehler beim auslesen des Tokens. Response: %', _response;
    END IF;

    RETURN _token;

  END $$ LANGUAGE plpgsql;


-- Funktion die den JWT-Token aus dem Cookie extrahiert und den Benutzer authentifiziert
CREATE OR REPLACE FUNCTION api_basic_auth.jwt_validate_from_cookie()
  RETURNS boolean
  SECURITY DEFINER
  AS $BODY$
  DECLARE
    vi_jwt_exp      integer;
    vi_jwt_session  varchar;
    vi_jwt_role     varchar;
    vi_cookie_token text;

    v_auth_token    jsonb;
    v_token_valid   boolean;
  BEGIN
    -- extract and parse jwt from cookie if cookie exists and set it as the jwt claim
    vi_cookie_token := current_setting( 'request.cookies', true )::json ->> 'jwt';
    IF vi_cookie_token IS NOT null
    THEN
      -- check token claims and validity
      SELECT payload     , valid
        INTO v_auth_token, v_token_valid
        FROM api_basic_auth.jwt_verify(
               token     => vi_cookie_token,
               secret    => TSystem.Settings__Database__Get('pgrst.jwt_secret'),
               algorithm => 'HS256');

      -- raise exception if token is not valid
      IF NOT v_token_valid
      THEN
        SELECT returnfailure('28000','validatesession','token is no longer valid');
      END IF;

      -- set token claims from cookie to jwt claims
      PERFORM set_config(
          'request.jwt.claims',
          v_auth_token::text,
          true
        );
      RETURN v_token_valid;
    ELSE
      RETURN false;
    END IF;
  END $BODY$ LANGUAGE plpgsql;


------------------------------------------------------------------------
-- F2 Funktionen
------------------------------------------------------------------------

-- Funktion um die F2-IDs für ein Modul und Feld zu ermitteln
--- Diese Funktion gibt die IDs der F2-Felder zurück, die für ein bestimmtes Modul und Feld relevant sind.
CREATE OR REPLACE FUNCTION API.f2__get_id(
      IN moduln varchar,
      IN feldn  varchar,
      IN kunde  varchar DEFAULT null,
      OUT f2_ids jsonb
    )
    RETURNS jsonb
    AS $$

    SELECT jsonb_agg(jsonb_build_object('f2_id', f2_id, 't_feld', trim(Concat(lang_text(vartxtnr), '   ', f2_kunde))::varchar ))
      FROM f2poss
      LEFT OUTER JOIN f2standard ON f2_standard = f2s_name
      LEFT OUTER JOIN f2order ON f2_stamp = f2o_f2_stamp AND f2order.insert_by = current_user AND NOT f2o_deact
      JOIN LATERAL (SELECT * FROM TSystem.f2poss__condition__get(f2_id)) AS cond ON TRUE -- #14961 (Feldinhaltabhängiges F2)
      WHERE (modulname ILIKE moduln
          OR (moduln ILIKE 'TFormEditTable-%' AND modulname ILIKE 'TFormEditTable')
          OR (moduln ILIKE 'TFrameF2Register-%' AND modulname ILIKE 'TFrameF2Register'))
        AND (feldname ILIKE feldn)
        AND (f2_kunde IS NULL OR f2_kunde LIKE coalesce(kunde, tsystem.settings__get('KUNDE')))
        AND NOT fastf2
        AND NOT f2_deleted
        AND (cond.field_name IS NULL AND cond.field_value IS NULL)
      --SQL wird mit Filter gefüllt. Siehe F2.dpr "Anzeige aller bearbeiteten F2-Fenster" (and f2_localmodified)
        AND NOT EXISTS (SELECT true FROM f2order WHERE f2o_f2_stamp=f2_stamp AND f2o_deact)--beim Kunden deaktivierte ausblenden

    $$ LANGUAGE sql;


-- Funktion um die F2-Rückfields für ein F2 zu ermitteln
-- gibt es keine Rückfields, dann wird der Feldname des RTF verwendet
CREATE OR REPLACE FUNCTION api.f2__rck_fields__get__by_f2_id(
    f2_id integer
  )
  RETURNS jsonb
  AS $$
    SELECT json_object_agg( coalesce( f2r_f2rckfieldn, feldname ), coalesce( f2r_forcomponent, feldname ) )
      FROM f2poss
      LEFT JOIN f2rck ON f2r_f2_id = f2poss.f2_id
      WHERE f2poss.f2_id = f2__rck_fields__get__by_f2_id.f2_id
  $$ PARALLEL SAFE LANGUAGE sql;

-- Funktion erstellt F2-Tabellen-HTML zu einer F2-ID
--- Diese Funktion generiert eine HTML-Tabelle aus den Daten, die von der F2-Query zurückgegeben werden.
CREATE OR REPLACE FUNCTION api.f2__result_table__get__as_html(
    f2_id integer,
    params jsonb DEFAULT '{}'::jsonb
  )
  RETURNS "text/html"
  AS $$
  DECLARE
    json_data jsonb[];
    html      text := '';
    row       jsonb;
    keys      text[];
    row_key   text;
    first_row jsonb;
  BEGIN

    -- 0. Params bereinigen
    IF NOT (params ? 'SearchEd') THEN
      params := params || jsonb_build_object( 'SearchEd', '%' );
    END IF;

    -- 1. F2-Ergebnis-Tabelle als JSON-Array laden
    SELECT array_agg(elem)
      INTO json_data
      FROM (
            SELECT jsonb_array_elements(
                     tsystem.sql_query__execute__to_jsonb_array(
                       concat_ws( E'\n', f2s_query, f2_query ),
                       params
                    )
                   ) AS elem
              FROM f2poss
              LEFT JOIN f2standard ON f2s_name = f2_standard
             WHERE f2poss.f2_id = f2__result_table__get__as_html.f2_id
            ) AS sub;

    IF array_length( json_data, 1 ) IS NULL THEN
      RETURN '<p>Keine Daten gefunden.</p>';
    END IF;

    -- 2. Keys aus der ersten Zeile ermitteln (für Tabellenüberschriften)
    first_row := json_data[1];
    keys := ARRAY( SELECT jsonb_object_keys( first_row ) );

    -- 3. HTML-Tabellenkopf aufbauen
    html := '<table class="min-w-full bg-white rounded-lg shadow">';
    html := html || '<thead class="bg-gray-100 sticky top-0"><tr>';
    FOREACH row_key IN ARRAY keys LOOP
      html := html || format('<th id=%I class="px-6 py-3 text-left text-sm font-semibold text-gray-600 bg-gray-100 select-none">%s</th>', row_key, TSystem.FieldAlias__Get(row_key));
    END LOOP;
    html := html || '</tr></thead><tbody>';

    -- 4. Zeilen ausgeben
    FOREACH row IN ARRAY json_data LOOP
      html := html || '<tr class="border-b cursor-pointer hover:bg-gray-50">';
      FOREACH row_key IN ARRAY keys LOOP
        html := html || format('<td class="px-2 py-1 text-sm">%s</td>', coalesce(row ->> row_key, ''));
      END LOOP;
      html := html || '</tr>';
    END LOOP;

    html := html || '</tbody></table>';

    RETURN html;
  END $$ LANGUAGE plpgsql;


------------------------------------------------------------------------
-- Offline-Komponenten
------------------------------------------------------------------------

-- ===== api.ping() =====
CREATE OR REPLACE FUNCTION api.ping()
  RETURNS void
  LANGUAGE sql
  IMMUTABLE
  AS $$
    SELECT;
  $$;

-- ===== api.manifest() =====
CREATE OR REPLACE FUNCTION api.manifest()
  RETURNS jsonb
  AS $$
    SELECT jsonb_build_object(
      'name', 'Prodat ERP Web App',
      'short_name', 'Prodat ERP',
      'start_url', 'login_page',
      'display', 'standalone',
      'background_color', '#ffffff',
      'theme_color', '#0d6efd',
      'icons', jsonb_build_array(
        jsonb_build_object(
          'src', './file?path=prodat_logo_pfeil.png',
          'sizes', '192x192',
          'type', 'image/png'
        )
      )
    );
  $$ LANGUAGE sql IMMUTABLE;


------------------------------------------------------------------
-- Wrapper-Funktionen für die API
------------------------------------------------------------------

-- Funktion um Standard-SQL-Abfragen auszuführen und das Ergebnis als JSONB zurückzugeben
-- Diese Funktion ist ein Wrapper um die tsystem.sql_query__execute__to_jsonb_set
-- und ermöglicht die Ausführung von Standard-SQL-Abfragen per API.
CREATE OR REPLACE FUNCTION api.execute_standard_sql_to_jsonb_set(
    standard_sql varchar,
    param_map jsonb DEFAULT NULL::jsonb
  )
  RETURNS SETOF jsonb
  AS $$

    SELECT * FROM tsystem.sql_query__execute__to_jsonb_set(
      tsystem.systemsqlstatement__query( standard_sql ), param_map );

  $$ LANGUAGE sql;

-- Funktion um Platzhalter der Form "xtt123" in einem Text durch die entsprechenden Sprachtexte zu ersetzen
-- Dabei wird die Funktion prodat_languages.lang_text() verwendet.
CREATE OR REPLACE FUNCTION api.lang_text__replace_xtt(
    p_text text
  )
  RETURNS text
  AS $$
  DECLARE
    v_result  text := p_text;
    v_match   text[];   -- [1] = kompletter Token (z.B. "xtt734" oder "XTT734"), [2] = Zahl
    v_token   text;
    v_num     int;
    v_repl    text;
  BEGIN
    -- Alle unterschiedlichen Treffer ermitteln; "m" ist EIN Array (eine Spalte)
    FOR v_match IN
      SELECT DISTINCT m
      FROM regexp_matches(p_text, E'(?i)(xtt)(\\d+)', 'g') AS t(m)
    LOOP
      v_token := v_match[1] || v_match[2];           -- vollständiger Token in Original-Schreibweise
      v_num   := v_match[2]::int;      -- nur die Zahl

      -- Auflösen; bei Fehlern/NULL: Token stehenlassen
      BEGIN
        v_repl := prodat_languages.lang_text(v_num);
        IF v_repl IS NULL THEN
          CONTINUE;
        END IF;
      EXCEPTION WHEN OTHERS THEN
        CONTINUE;
      END;

      -- Sämtliche Vorkommen dieses Tokens ersetzen (genau in der gefundenen Schreibweise)
      v_result := replace(v_result, v_token, v_repl);
    END LOOP;

    RETURN v_result;
  END $$ LANGUAGE plpgsql strict;